<?php
session_start();
// Define o fuso horário para garantir que as datas sejam consistentes
date_default_timezone_set('America/Sao_Paulo'); // Define para o fuso horário de Recife/Brasil

// Caminho para o banco de dados padrão
$db_file = 'api/.fzstoredev.db';

// Conexão com o banco de dados SQLite
try {
    $pdo = new PDO("sqlite:" . $db_file);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);

    // Garante que a tabela 'usuarios' seja criada se ainda não existir
    $pdo->exec("CREATE TABLE IF NOT EXISTS usuarios (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT UNIQUE NOT NULL,
        password TEXT NOT NULL
    )");

} catch (PDOException $e) {
    die("Erro ao conectar ou criar o banco de dados: " . $e->getMessage());
}

$success = false; // Variável para controlar o sucesso do registro
$erro = ""; // Inicializa a variável de erro
$registro_concluido = false; // Flag para indicar se o registro foi bem-sucedido

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $username = trim($_POST["username"]);
    $password = trim($_POST["password"]);
    $confirm_password = trim($_POST["confirm_password"]);

    // Validação básica
    if (empty($username) || empty($password) || empty($confirm_password)) {
        $erro = "Todos os campos são obrigatórios!";
    } elseif ($password !== $confirm_password) {
        $erro = "As senhas não coincidem!";
    } elseif (strlen($password) < 6) {
        $erro = "A senha deve ter pelo menos 6 caracteres!";
    } else {
        // Verifica se o nome de usuário já existe
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM usuarios WHERE username = ?");
        $stmt->execute([$username]);
        if ($stmt->fetchColumn() > 0) {
            $erro = "Nome de usuário já existe. Escolha outro.";
        } else {
            // Hash da senha antes de armazenar
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);

            // Insere o novo usuário no banco de dados
            try {
                $stmt = $pdo->prepare("INSERT INTO usuarios (username, password) VALUES (?, ?)");
                $stmt->execute([$username, $hashed_password]);
                $success = true;
                $registro_concluido = true;
            } catch (PDOException $e) {
                $erro = "Erro ao registrar o usuário: " . $e->getMessage();
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cadastro</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
    <style>
        :root {
            /* Cores base Material Design 3 inspired */
            --md-sys-color-primary: #6750A4;
            --md-sys-color-on-primary: #FFFFFF;
            --md-sys-color-primary-container: #EADDFF;
            --md-sys-color-on-primary-container: #21005D;
            --md-sys-color-secondary: #625B71;
            --md-sys-color-on-secondary: #FFFFFF;
            --md-sys-color-secondary-container: #E8DEF8;
            --md-sys-color-on-secondary-container: #1D192B;
            --md-sys-color-tertiary: #7D5260;
            --md-sys-color-on-tertiary: #FFFFFF;
            --md-sys-color-tertiary-container: #FFD8E4;
            --md-sys-color-on-tertiary-container: #31111D;
            --md-sys-color-error: #BA1A1A;
            --md-sys-color-on-error: #FFFFFF;
            --md-sys-color-error-container: #FFDAD6;
            --md-sys-color-on-error-container: #410002;
            --md-sys-color-background: #FFFBFE;
            --md-sys-color-on-background: #1C1B1F;
            --md-sys-color-surface: #FFFBFE;
            --md-sys-color-on-surface: #1C1B1F;
            --md-sys-color-surface-variant: #E7E0EC;
            --md-sys-color-on-surface-variant: #49454F;
            --md-sys-color-outline: #79747E;
            --md-sys-color-shadow: #000000;
            --md-sys-color-inverse-surface: #313033;
            --md-sys-color-inverse-on-surface: #F4EFF4;
            --md-sys-color-inverse-primary: #D0BCFF;
            --md-sys-color-surface-tint: #6750A4;
        }

        body {
            margin: 0;
            padding: 0;
            background: linear-gradient(145deg, #1A1C20, #2C2F33, #3E4247);
            color: #E8DEF8;
            font-family: 'Roboto', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100vh;
        }

        .register-container { /* Nome da classe alterado para registro */
            background-color: #313033;
            padding: 40px 30px;
            border-radius: 28px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2),
                        0 8px 24px rgba(0, 0, 0, 0.3);
            width: 340px;
            text-align: center;
            box-sizing: border-box;
        }

        .register-container img.logo {
            max-width: 130px;
            margin-bottom: 25px;
            filter: drop-shadow(0 2px 5px rgba(0, 0, 0, 0.3));
        }

        .register-container h2 {
            margin-bottom: 30px;
            color: var(--md-sys-color-primary-container);
            font-size: 2rem;
            letter-spacing: 0.01em;
            font-weight: 500;
        }

        .register-container input {
            width: calc(100% - 30px);
            padding: 14px 15px;
            margin-bottom: 20px;
            border: 1px solid var(--md-sys-color-outline);
            border-radius: 12px;
            background-color: #49454F;
            color: var(--md-sys-color-on-surface);
            transition: border-color 0.3s, box-shadow 0.3s;
            font-size: 1rem;
            box-sizing: border-box;
        }

        .register-container input::placeholder {
            color: var(--md-sys-color-on-surface-variant);
            opacity: 0.7;
        }

        .register-container input:focus {
            border-color: var(--md-sys-color-primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(103, 80, 164, 0.2);
            background-color: #55505C;
        }

        .register-container button {
            width: 100%;
            padding: 15px;
            border: none;
            border-radius: 20px;
            color: var(--md-sys-color-on-primary);
            background: var(--md-sys-color-primary);
            font-weight: 500;
            font-size: 1.1rem;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1),
                        0 4px 8px rgba(0, 0, 0, 0.2);
            cursor: pointer;
            transition: background-color 0.3s, box-shadow 0.3s, transform 0.2s;
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 8px;
        }

        .register-container button:hover {
            background-color: #7A66C0;
            transform: translateY(-1px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2),
                        0 6px 12px rgba(0, 0, 0, 0.3);
        }

        .register-container button:active {
            background-color: #55418F;
            transform: translateY(0);
            box-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
        }

        .error-message {
            margin-top: 20px;
            background-color: var(--md-sys-color-error-container);
            padding: 12px;
            border-radius: 8px;
            color: var(--md-sys-color-on-error-container);
            font-weight: 500;
            font-size: 0.95rem;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        }

        .login-link { /* Nova classe para o link de login */
            margin-top: 20px;
            display: inline-block;
            color: var(--md-sys-color-secondary-container);
            text-decoration: none;
            font-size: 0.95rem;
            transition: color 0.3s;
        }
        .login-link:hover {
            color: var(--md-sys-color-primary-container);
        }
    </style>
</head>
<body>

<div class="register-container">
    <img src="https://i.imgur.com/JJ1GKoQ.png" alt="Logo HL" class="logo">
    <h2><i class="fas fa-user-plus"></i> CADASTRAR</h2>
    <form method="POST">
        <label for="username" style="position: absolute; left: -9999px;">Usuário</label>
        <input type="text" name="username" id="username" placeholder="Usuário" required autocomplete="new-username">
        
        <label for="password" style="position: absolute; left: -9999px;">Senha</label>
        <input type="password" name="password" id="password" placeholder="Senha" required autocomplete="new-password">
        
        <label for="confirm_password" style="position: absolute; left: -9999px;">Confirmar Senha</label>
        <input type="password" name="confirm_password" id="confirm_password" placeholder="Confirmar Senha" required autocomplete="new-password">
        
        <button type="submit"><i class="fas fa-user-plus"></i> Cadastrar</button>
    </form>
    <?php if (!empty($erro)): ?>
        <div class="error-message"><?= htmlspecialchars($erro, ENT_QUOTES, 'UTF-8') ?></div>
    <?php endif; ?>

    <a href="login.php" class="login-link">Já tem uma conta? Faça login aqui!</a>
</div>

<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
    // Usar PHP para controlar a exibição do SweetAlert2
    <?php if ($registro_concluido): ?>
        Swal.fire({
            icon: 'success',
            title: 'Cadastro Realizado!',
            text: 'Sua conta foi criada com sucesso. Você será redirecionado para a tela de login.',
            timer: 3000, // Exibe por 3 segundos
            timerProgressBar: true,
            showConfirmButton: false,
            allowOutsideClick: false,
            didOpen: () => {
                Swal.showLoading();
            },
            willClose: () => {
                window.location.href = 'login.php'; // Redireciona para login.php após o timer
            }
        });
    <?php endif; ?>

    // Se houver erro, exibe o SweetAlert2 com a mensagem de erro
    <?php if (!empty($erro)): ?>
        Swal.fire({
            icon: 'error',
            title: 'Erro no Cadastro!',
            text: '<?= htmlspecialchars($erro, ENT_QUOTES, 'UTF-8') ?>',
            confirmButtonText: 'Tentar Novamente',
            customClass: {
                confirmButton: 'swal2-confirm-button-custom'
            }
        });
    <?php endif; ?>
</script>

</body>
</html>